-- !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
-- !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
--
-- DO NOT CREATE functions, that take full path filenames as parameters !!!
--
--   * every function is callable by every user that can connect to the db
--   * which means every functions needs to be safe to be called
--   * having a function that takes a full path filename as parameter is a severe security RISK
--   * it would allow every user to actually write everywhere to the disk of the pg-server
--
-- !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!



--------------------------------------------------------------------------------
-- !!! NOT supposed to be called by users directly !!!
-- helper for python functions
CREATE OR REPLACE FUNCTION tlog.dblog__raisenotice(a_message varchar)
  RETURNS void
  AS $$
  BEGIN
    RAISE NOTICE '%', a_message;
  END
  $$ LANGUAGE plpgsql;



--------------------------------------------------------------------------------
-- to fix the problem with having lots of functions that are not supposed to be called by users directly
--
-- CREATE ROLE <dblog_restricted_user> -- no rights except Sys.Prodat-User
--
-- for all function that should NOT be directyl called
--   ALTER FUNCTION xyz OWNER TO dblog_restricted_user;
--   GRANT EXECUTE ON FUNCTION xyz TO dblog_restricted_user;
--   REVOKE ALL ON FUNCTION xyz FROM PUBLIC;
--   REVOKE ALL ON FUNCTION xyz FROM "SYS.Prodat-User";
--
-- for all function that ARE supposed to be called by users directly
--   define them as: SECURITY DEFINER
--   ALTER FUNCTION xyz OWNER TO dblog_restricted_user;
--   GRANT EXECUTE ON FUNCTION xyz TO dblog_restricted_user;
--   GRANT EXECUTE ON FUNCTION xyz TO SYS.Prodat-User;



--------------------------------------------------------------------------------
-- create the filename in python(via calling a function) and with parameters would be the optimal/safe variant
-- this way no function would take a full path filename as parameter
-- and thus is safer (we DO WRITE to filesystem here, within the db-server ...)
-- configurate access right in a way that only the dblog schema functions can execute this one??

--------------------------------------------------------------------------------
-- !!! NOT supposed to be called by users directly !!!
-- gets dir for csvlogfile
CREATE OR REPLACE FUNCTION tlog.dblog__logfile__dir_name__get() RETURNS varchar AS
  $$
    import os
    import sys

    # -- Das Lesen des "data_directory" braucht Superuser-Berechtigungen.
    # -- Daher wird die Funktion mittels SECURITY DEFINER als Benutzer postgres ausgeführt.
    # --
    # -- Durch Benutzung von SD (Funktions-spezifisches Dictionary) Performance
    # -- auch bei Aufruf aus anderer Python-Funktion gegeben: tlog.dblog__logfile__log_line__write(a_csvline varchar)

    if 'dblog_directory' not in SD:
      _ret = plpy.execute("SELECT current_setting('data_directory') AS datadir, current_setting('dynamic_shared_memory_type') AS os, current_setting('logging_collector') AS logactive, COALESCE(current_setting('log_directory'), '') AS logsubdir")
      _log_dir = _ret[0]["datadir"];
      _os = _ret[0]["os"];
      _log_active = _ret[0]["logactive"];
      _log_subdir = _ret[0]["logsubdir"];

      if (_os == 'windows'):
        if (_log_active):
          if (_log_subdir != ''):
            _log_dir = os.path.join(_log_dir, _log_subdir)

      _log_dir = os.path.join(_log_dir, 'dblog')
      _log_dir = _log_dir.replace(os.sep, '/')

      if (not os.path.exists(_log_dir)):
        os.makedirs(_log_dir)

      SD['dblog_directory'] = _log_dir

    return SD['dblog_directory']
  $$ LANGUAGE plpython3u SECURITY DEFINER;

--------------------------------------------------------------------------------
-- force subdirectories
SELECT tlog.dblog__logfile__dir_name__get();

--------------------------------------------------------------------------------
-- !!! NOT supposed to be called by users directly !!!
-- transforms time-of-day into a staggered(_interval_minutes) number
CREATE OR REPLACE FUNCTION tlog.dblog__time_partition__get(a_time time DEFAULT now()::time)
  RETURNS int
  AS $$
  DECLARE
    _interval_minutes int;

  BEGIN
    _interval_minutes := 10;
    -- 00:00 ->    0
    -- 00:01 ->    0
    --    ...
    -- 00:09 ->    0
    -- 00:10 ->   10
    -- 00:11 ->   10
    -- ...
    -- 23:59 -> 1430

    RETURN (Trunc(Trunc(EXTRACT(EPOCH FROM a_time) / 60) / _interval_minutes) * _interval_minutes)::int;
    -- (Trunc(Trunc(EXTRACT(EPOCH FROM ( '23:59:00 UTC'::time ) ) / 60) / 10) * 10)::int
  END;
  $$ LANGUAGE plpgsql IMMUTABLE;

--------------------------------------------------------------------------------
-- !!! NOT supposed to be called by users directly !!!
-- reverse of tlog.dblog__time_partition__get
CREATE OR REPLACE FUNCTION tlog.dblog__time_from_partitioned__get(a_time_partitioned integer)
  RETURNS time
  AS $$
  BEGIN
    RETURN (TIME '00:00' + (a_time_partitioned * INTERVAL '1 minute'))::time;
  END;
  $$ LANGUAGE plpgsql IMMUTABLE;

--------------------------------------------------------------------------------
-- !!! NOT supposed to be called by users directly !!!
-- get just the filename part of the csvlogfile
--   # does not use anything beside alphanum + underscorem + dot, to enable very simple(fast) sanitizing of it
--     within tlog.dblog__logfile__log_line__write is possible
CREATE OR REPLACE FUNCTION tlog.dblog__logfile__file_name__get(
    a_date     date    DEFAULT today(),
    a_time     time    DEFAULT now()::time,
    a_pid      integer DEFAULT pg_backend_pid(),
    a_database varchar default current_database()
  )
  RETURNS varchar
  AS $$
    -- make sure it does match the filename masks used in
    --   tlog.dblog__logfiles__import_intern
    --   tlog.dblog__logfile__list
  BEGIN
    IF char_length( translate( a_database, '.-_0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz', '') ) <> 0 THEN
      RAISE WARNING 'Database name contains forbidden characters!';
    END IF;

    RETURN concat('csvlog.', a_database, '.', to_char(a_date, 'YYYYMMDD'), '.', to_char(tlog.dblog__time_partition__get(a_time), 'fm0000'), '.', a_pid::varchar, '.csv');
  END;
  $$ LANGUAGE plpgsql IMMUTABLE;

--------------------------------------------------------------------------------
-- !!! NOT supposed to be called by users directly !!!
-- gets full path including filename of csvlogfile
CREATE OR REPLACE FUNCTION tlog.dblog__logfile__full_path__get(
    a_date     date    DEFAULT today(),
    a_time     time    DEFAULT now()::time,
    a_pid      integer DEFAULT pg_backend_pid(),
    a_database varchar default current_database()
  )
  RETURNS varchar AS
  $$
    SELECT concat(tlog.dblog__logfile__dir_name__get(), '/', tlog.dblog__logfile__file_name__get(a_date, a_time, a_pid, a_database))
  $$ LANGUAGE sql STABLE;

--------------------------------------------------------------------------------
-- check if a csvlogfile does exist
-- while not realy supposed to be called by users directly, its allowable
CREATE OR REPLACE FUNCTION tlog.dblog__logfile__exists(
    a_date date DEFAULT today(),
    a_time time DEFAULT now()::time,
    a_pid integer DEFAULT pg_backend_pid(),
    a_database varchar default current_database()
  )
  RETURNS boolean AS
  $$
    import os

    _plan = plpy.prepare("SELECT tlog.dblog__logfile__full_path__get($1::date, $2::time, $3::integer, $4::varchar) AS filename", ["date","time","integer","varchar"])
    _ret = plpy.execute(_plan, [a_date, a_time, a_pid, a_database])
    _filename = _ret[0]["filename"]

    if os.path.exists(_filename):
      return True
    else:
      return False
  $$ LANGUAGE plpython3u;

--------------------------------------------------------------------------------
-- !!! NOT supposed to be called by users directly !!!
-- deletes a csvlogfile from disk
CREATE OR REPLACE FUNCTION tlog.dblog__logfile__delete(
    a_date date DEFAULT today(),
    a_time time DEFAULT now()::time,
    a_pid integer DEFAULT pg_backend_pid(),
    a_database varchar default current_database()
  )
  RETURNS void AS
  $$
    import os

    _plan = plpy.prepare("SELECT tlog.dblog__logfile__full_path__get($1::date, $2::time, $3::integer, $4::varchar) AS filename", ["date","time","integer","varchar"])
    _ret = plpy.execute(_plan, [a_date, a_time, a_pid, a_database])
    _filename = _ret[0]["filename"]

    _plan_notice = plpy.prepare("SELECT tlog.dblog__raisenotice($1::varchar);", ["varchar"])
    _message = "tlog.dblog__logfile__delete: " + _filename
    plpy.execute(_plan_notice, [_message])

    if os.path.exists(_filename):
      os.remove(_filename)
  $$ LANGUAGE plpython3u;

--------------------------------------------------------------------------------
-- !!! NOT supposed to be called by users directly !!!
-- helper function to determine if its ok to delete a csvlogfile
--   # accurate ONLY during the top-level call tlog.dblog__logfiles__import
--   # sessionvar's set: 'dblog_import_tx_id', 'dblog_import_time'
CREATE OR REPLACE FUNCTION tlog.dblog__logfile__candelete(row_data tlog.dblogfiles_imported)
  RETURNS boolean
  AS $$
  DECLARE
    _ret boolean;

  BEGIN
    _ret :=
    (
      -- only delete files which we imported right now
      (row_data.dblfi_txid = TSystem.SessionVar__get_varchar('dblog_import_tx_id'))
      AND
      (
        -- everything older than today() is definately no longer written to
        (today() > row_data.dblfi_date)
        OR
        -- everything with an older time(partition) of today is definately no longer written to
        (
          (today() = row_data.dblfi_date)
          AND
          (TSystem.SessionVar__get_integer('dblog_import_time') > row_data.dblfi_time)
        )
        OR
        -- our own session, noone else will write to it and deleting is only done from top-level calls (outside tx)
        -- inside top-level call, between the import(after) and the delete, there should never be any more entries written to the csvlogfile
        (row_data.dblfi_pid = pg_backend_pid())
        OR
        -- NOT our own session
        -- do not delete csvlogfiles which might be written to (active session with pid exists)
        (
          NOT(row_data.dblfi_pid = pg_backend_pid())
          AND (NOT(EXISTS(SELECT true FROM pg_stat_activity WHERE (pid = row_data.dblfi_pid))))
        )
      )
    );

    RAISE NOTICE 'candelete: % %', row_data, _ret;

    return _ret;
  END;
  $$ LANGUAGE plpgsql;

--------------------------------------------------------------------------------
-- !!! NOT supposed to be called by users directly !!!
-- delete csvlogfiles from disk, which have an entry(flag) in tlog.dblogfiles_imported
CREATE OR REPLACE FUNCTION tlog.dblog__logfile__import__cleanup_stage2()
  RETURNS void
  AS $$
  BEGIN
    PERFORM
      tlog.dblog__logfile__delete(dblfi_date, tlog.dblog__time_from_partitioned__get(dblfi_time), dblfi_pid, dblfi_database)
    FROM
      tlog.dblogfiles_imported
    WHERE
      tlog.dblog__logfile__candelete(tlog.dblogfiles_imported.*)
    ;
  END;
  $$ LANGUAGE plpgsql;

--------------------------------------------------------------------------------
-- !!! NOT supposed to be called by users directly !!!
-- delete entries(flags) in table tlog.dblogfiles_imported,
--   # accurate ONLY during the top-level call tlog.dblog__logfiles__import
--   # which do not have a corresponding csvlogfile on disk
--   # or where imported right now
CREATE OR REPLACE FUNCTION tlog.dblog__logfile__import__cleanup_stage1()
  RETURNS void
  AS $$
  BEGIN
    DELETE FROM
      tlog.dblogfiles_imported
    WHERE
      (
        -- flags for nonexisting files need to be removed
        (NOT(tlog.dblog__logfile__exists(dblfi_date, tlog.dblog__time_from_partitioned__get(dblfi_time), dblfi_pid, dblfi_database)))
        OR
        -- cleanup all flags for the current import
        (dblfi_txid = TSystem.SessionVar__get_varchar('dblog_import_tx_id'))
      )
    ;
  END;
  $$ LANGUAGE plpgsql;

--------------------------------------------------------------------------------
-- !!! NOT supposed to be called by users directly !!!
-- manually import a single csvlogfile into tlog.dblog
--   # import file + add to table dblog.csvlogfiles
--   # if up-to-date logs are needed during a transaction,
--     this function can be called, but csvlogfile will not be deleted
--     better to call tlog.dblog__import_sessionlogs, because of partitioned time in filename
--   # importing the same csvlogfile over and over is ok, only new entries will be actually imported
CREATE OR REPLACE FUNCTION tlog.dblog__logfile__import(
    a_date     date    DEFAULT today(),
    a_time     time    DEFAULT now()::time,
    a_pid      integer DEFAULT pg_backend_pid(),
    a_database varchar default current_database()
  )
  RETURNS void
  SET escape_string_warning = false
  AS $$
  DECLARE
    _logfilename varchar;
    _temptable   varchar;
    _SQL         varchar;

  BEGIN
    IF (a_database = current_database()) THEN
      IF ( tlog.dblog__logfile__exists(a_date, a_time, a_pid, a_database) ) THEN
        SELECT
          tlog.dblog__logfile__full_path__get(a_date, a_time, a_pid, a_database)
        INTO
          _logfilename
        ;

        _temptable := FORMAT('dblog_tmp_%s_%s', to_char(today(), 'YYYYMMDD'), pg_backend_pid()::varchar);
        EXECUTE FORMAT('DROP TABLE IF EXISTS %s',                            _temptable);
        EXECUTE FORMAT('CREATE TEMP TABLE %s(dbl_id bigint, logentry json)', _temptable);

        -- QUOTE e'\x01' DELIMITER e'\x02
        -- This is needed, to allow json-escaped characters inside a json-string
        -- linebreaks are escaped in json-strings with \n
        _SQL := FORMAT('COPY %s FROM ''%s'' CSV QUOTE e''\x01'' DELIMITER e''\x02'';', _temptable, _logfilename);
        EXECUTE _SQL;

        _SQL :=
          FORMAT
          (
            $SQL$
              WITH src AS
              (
                SELECT
                  jsonrow.*
                FROM
                  %s AS tmptable
                  CROSS JOIN json_populate_record(NULL::tlog.dblog, tmptable.logentry) AS jsonrow
              )
              INSERT INTO
                tlog.dblog
              SELECT
                *
              FROM
                src
              WHERE
                NOT EXISTS(SELECT dbl_id FROM tlog.dblog WHERE dbl_id = src.dbl_id)
              ;
            $SQL$,
            _temptable
          )
        ;
        EXECUTE _SQL;

        INSERT INTO
          tlog.dblogfiles_imported (dblfi_date, dblfi_time, dblfi_pid, dblfi_database, dblfi_txid)
        VALUES
          (a_date, tlog.dblog__time_partition__get(a_time), a_pid, a_database, pg_current_xact_id_if_assigned()::varchar)
        ON CONFLICT DO NOTHING
        ;

        RAISE NOTICE 'tlog.dblog__logfile__import ok: % (% % % % %)', _logfilename, to_char(a_date, 'YYYYMMDD'), to_char(a_time, 'HH24MI'), a_database, a_pid, pg_current_xact_id_if_assigned()::varchar;
      ELSE
        RAISE NOTICE 'tlog.dblog__logfile__import: does not exists or already imported %, %, %, %', to_char(a_date, 'YYYYMMDD'), to_char(a_time, 'HH24MI'), a_pid, a_database;
      END IF;
    ELSE
      RAISE WARNING 'tlog.dblog__logfile__import: wrong database %, %, %, %', to_char(a_date, 'YYYYMMDD'), to_char(a_time, 'HH24MI'), a_pid, a_database;
    END IF;
  END;
  $$ LANGUAGE plpgsql SECURITY DEFINER;

--------------------------------------------------------------------------------
-- !!! NOT supposed to be called by users directly !!!
-- manually import csvlogfiles into tlog.dblog
--   # iterates log-directory through for csvlogfiles
--   # pass restrictprocessingto = -1 to force import of all existing csvlogfiles, otherwise its the max number of csvlogfiles to import
CREATE OR REPLACE FUNCTION tlog.dblog__logfiles__import_intern(
    currentsession       boolean,
    othersessions        boolean,
    thisday              boolean,
    olderthanthisday     boolean,
    restrictprocessingto integer DEFAULT 10,
    recursion            integer DEFAULT 0
  )
  RETURNS integer AS
  $$
    import sys
    import glob
    import datetime

    _indent = ''.ljust(recursion * 2)

    _plan_notice = plpy.prepare("SELECT tlog.dblog__raisenotice($1::varchar);", ["varchar"])
    _plan_sub = plpy.prepare("SELECT tlog.dblog__logfiles__import_intern($1::boolean, $2::boolean, $3::boolean, $4::boolean, $5::integer, $6::integer) AS count_processed;", ["boolean","boolean","boolean","boolean","integer","integer"])

    _ret = plpy.execute("SELECT pg_backend_pid() AS pid, current_database() AS db, tlog.dblog__logfile__dir_name__get() AS logdir")
    _current_pid = _ret[0]["pid"]
    _current_db = _ret[0]["db"]
    _LogDir = _ret[0]["logdir"] + '/'
    # -- [DEBUG]_LogDir = "C:/Users/PRODAT~1/AppData/Local/Temp/prodat/pg/dblog/alt/"
    _LenDir = len(_LogDir)

    _message = "{}tlog.dblog__logfiles__import_intern: PID:{} CS:{} OS:{} TD:{} OD:{} Allowed:{}".format(_indent, _current_pid, currentsession, othersessions, thisday, olderthanthisday, restrictprocessingto)
    plpy.execute(_plan_notice, [_message])

    _count_processed = 0
    if (not(restrictprocessingto == -1)):
      # -- restrict the number of files to process
      # -- check if we can divide the call into several calls, prioritizing logs:
      # --   for the current session
      # --   for this day
      # --   for the most current

      _max_allowed_count = restrictprocessingto

      if (currentsession and othersessions):
        # -- first process logs for this session
        _ret = plpy.execute(_plan_sub, [True, False, thisday, olderthanthisday, _max_allowed_count, recursion + 1])
        _count_processed = _count_processed + _ret[0]["count_processed"]
        _max_allowed_count = _max_allowed_count - _count_processed
        if (_max_allowed_count > 0):
          # -- if we did not exceed the allowed number of files to process, now process other sessions
          _ret = plpy.execute(_plan_sub, [False, True, thisday, olderthanthisday, _max_allowed_count, recursion + 1])
          _count_processed = _count_processed + _ret[0]["count_processed"]

        # -- return how many were processed (early return, no need to to the extra stuff of scanning for files)
        return _count_processed
      elif (not(currentsession and othersessions) and (thisday and olderthanthisday)):
        # -- first process logs for today
        _ret = plpy.execute(_plan_sub, [currentsession, othersessions, True, False, _max_allowed_count, recursion + 1])
        _count_processed = _count_processed + _ret[0]["count_processed"]
        _max_allowed_count = _max_allowed_count - _count_processed
        if (_max_allowed_count > 0):
          # -- if we did not exceed the allowed number of files to process, now process other older than this day
          _ret = plpy.execute(_plan_sub, [currentsession, othersessions, False, True, _max_allowed_count, recursion + 1])
          _count_processed = _count_processed + _ret[0]["count_processed"]

        # -- return how many were processed (early return, no need to to the extra stuff of scanning for files)
        return _count_processed
    else:
      # -- no restrictions regarding how many files we do process
      _max_allowed_count = sys.maxsize

    _plan_convert = plpy.prepare("SELECT tlog.dblog__time_from_partitioned__get($1::integer) as time;", ["integer"])
    _plan_import = plpy.prepare("SELECT tlog.dblog__logfile__import($1::date, $2::time, $3::integer, $4::varchar) AS filename", ["date","time","integer","varchar"])

    _message = "{} #checking: {}".format(_indent, _LogDir)
    plpy.execute(_plan_notice, [_message])

    # -- csvlog . db-name . date(iso) . time(partitioned/number) . pid . csv
    # -- 0        1         2           3                          4     5
    # -- this will sort the files in descending order regarding date/time
    # -- allowing us to prioritize processing of the most current
    _files_sorted = sorted(glob.glob(_LogDir + "csvlog.*.*.*.*.csv"), reverse=True)
    for _file in _files_sorted:
      if (_count_processed >= _max_allowed_count):
        break

      _filename = _file[_LenDir:]
      _parts = _filename.split(".")
      _partscount = len(_parts)

      if ((_partscount == 6) and (_parts[0] == 'csvlog') and (_parts[5] == 'csv')):
        _database = _parts[1]

        if (_database == _current_db):
          # -- [DEBUG] message here after this if to reduce amount of notices
          # --_message = "{} #found: {}".format(_indent, _filename)
          # --plpy.execute(_plan_notice, [_message])

          try:
            _pid = int(_parts[4])
          except Exception as _E:
            _pid = 0

          try:
            _date = datetime.datetime.strptime(_parts[2], '%Y%m%d')
          except Exception as _E:
            _pid = 0

          try:
            _time_partitioned = int(_parts[3])
            _ret = plpy.execute(_plan_convert, [_time_partitioned])
            _time = _ret[0]["time"]
          except Exception as _E:
            _pid = 0

          _doimport_date = False
          _doimport_pid = False

          if (currentsession and (_pid == _current_pid)):
            _doimport_pid = True
          if (othersessions and (_pid != _current_pid)):
            _doimport_pid = True

          _diff = datetime.datetime.now() - _date
          if (thisday and (_diff.days == 0)):
            _doimport_date = True
          if (olderthanthisday and (_diff.days > 0)):
            _doimport_date = True

          if ((_pid > 0) and _doimport_date and _doimport_pid):
            # -- [DEBUG]
            # --_message = "{} #processing: {}".format(_indent, _filename)
            # --plpy.execute(_plan_notice, [_message])
            _ret = plpy.execute(_plan_import, [_date, _time, _pid, _database])
            _count_processed = _count_processed + 1

    return _count_processed
  $$ LANGUAGE plpython3u;

-- !!! this is the correct method/procedure to use !!!
--   # its a PROCEDURE, so it must be called as top-level statement: CALL tlog.dblog__logfiles__import_and_cleanup(...)
--   # pass restrictprocessingto = -1 to force import of all existing csvlogfiles otherwise its the max number of csvlogfiles to import
CREATE OR REPLACE PROCEDURE tlog.dblog__logfiles__import_and_cleanup(
    currentsession       boolean,
    othersessions        boolean,
    thisday              boolean,
    olderthanthisday     boolean,
    restrictprocessingto integer DEFAULT 10
  )
  AS $$
  DECLARE
    _time time;

  BEGIN
    -- ! A SECURITY DEFINER procedure cannot execute transaction control statements !

    -- this will throw an exception, IF we are already in an transaction initiated from outside
    COMMIT;

    -- so this was actually called as top-level statement without any transaction already initiated

    --
    _time := now()::time;

    -- import and flag as imported
    BEGIN
      -- force generation of tx_id
      INSERT INTO tlog.dblog
        (dbl_type,       dbl_level,  dbl_source, dbl_ctx,                              dbl_msg)
      VALUES
        ('pltDebugging', 'pllDebug', 'plsDB',    'dblog__logfiles__import_outside_tx', 'start')
      ;

      -- log tx_id (implicit)
      INSERT INTO tlog.dblog
        (dbl_type,       dbl_level,  dbl_source, dbl_ctx,                              dbl_msg)
      VALUES
        ('pltDebugging', 'pllDebug', 'plsDB',    'dblog__logfiles__import_outside_tx', 'txid')
      ;

      -- set session var's for __cleanup_stage / __candelete
      PERFORM TSystem.SessionVar__set_varchar('dblog_import_tx_id', coalesce(pg_current_xact_id_if_assigned()::varchar, ''), false);
      PERFORM TSystem.SessionVar__set_integer('dblog_import_time', tlog.dblog__time_partition__get(_time),                   false);
      RAISE NOTICE 'tlog.dblog__logfiles__import: tx = % pid = % time = %(%)', pg_current_xact_id_if_assigned()::varchar, pg_backend_pid(),  tlog.dblog__time_partition__get(_time), _time;

      -- import and flag
      PERFORM tlog.dblog__logfiles__import_intern(currentsession, othersessions, thisday, olderthanthisday, restrictprocessingto);
    end;
    COMMIT;

    -- remove files which were flaged(imported) and are safe to delete
    BEGIN
      PERFORM tlog.dblog__logfile__import__cleanup_stage2();
    END;
    COMMIT;

    -- remove flag for nonexisting files and those from this import
    BEGIN
      PERFORM tlog.dblog__logfile__import__cleanup_stage1();
    END;
    COMMIT;
  END;
  $$ LANGUAGE plpgsql;

--------------------------------------------------------------------------------
-- convenience function, not supposed to be called, rather use tlog.dblog__logfiles__import_and_cleanup directly
--   # interims solution
--   # this one MUST be removed, because it has hardcoded credentials
--   # pass restrictprocessingto = -1 to force import of all existing csvlogfiles otherwise its the max number of csvlogfiles to import
CREATE OR REPLACE FUNCTION tlog.dblog__logfiles__import(
    currentsession       boolean,
    othersessions        boolean,
    thisday              boolean,
    olderthanthisday     boolean,
    restrictprocessingto integer DEFAULT 10
  )
  RETURNS void
  AS $$
  DECLARE
    _dblink          varchar;
    _sql             varchar;

  BEGIN
    --- _dblink := FORMAT('host=localhost port=%s dbname=%s user=postgres password=sysdba', (inet_server_port()::integer)::varchar, current_database()::varchar);
    SELECT tsystem.dblink__connectionstring__get() INTO _dblink;

    _sql := FORMAT('CALL tlog.dblog__logfiles__import_and_cleanup(%s, %s, %s, %s, %s)', currentsession::text, othersessions::text, thisday::text, olderthanthisday::text, restrictprocessingto::varchar);

    PERFORM * FROM dblink(_dblink, _sql) AS sub (dummy text);
  END;
  $$ LANGUAGE plpgsql;

--------------------------------------------------------------------------------
-- convenience function to import the csvlogfiles for the current session
--   # for up to now()
--   # but not older than 1 staggered csvlogfile before the current one
--   # does NOT delete any csvlogfiles
CREATE OR REPLACE FUNCTION tlog.dblog__import_sessionlogs() RETURNS void
  AS $$
  DECLARE
    _date date;
    _time time;
    _tp   integer;

  BEGIN
    _date := today();
    _time := now()::time;
    _tp   := tlog.dblog__time_partition__get(_time);

    RAISE NOTICE 'dblog__import_sessionlogs: % % %', _date, _time, _tp;

    IF (_tp > 0) THEN
      -- import the logfile to that was written to up to until _time
      PERFORM tlog.dblog__logfile__import(_date, tlog.dblog__time_from_partitioned__get(_tp    ), pg_backend_pid()::integer, current_database()::varchar);
      -- import the logfile previously written to
      PERFORM tlog.dblog__logfile__import(_date, tlog.dblog__time_from_partitioned__get(_tp - 1), pg_backend_pid()::integer, current_database()::varchar);
    ELSE
      -- import the logfile to that was written to up to until _time
      PERFORM tlog.dblog__logfile__import(_date, tlog.dblog__time_from_partitioned__get(0), pg_backend_pid()::integer, current_database()::varchar);

      -- import the logfile previously written to
      _time := (_date::TIMESTAMP - INTERVAL '1 second')::time;
      _date := (_date::TIMESTAMP - INTERVAL '1 second')::date;
      PERFORM tlog.dblog__logfile__import(_date, _time, pg_backend_pid()::integer, current_database()::varchar);
    END IF;

    -- cleanup flags set while importing
    PERFORM TSystem.SessionVar__set_varchar('dblog_import_tx_id', coalesce(pg_current_xact_id_if_assigned()::varchar, ''), false);
    PERFORM tlog.dblog__logfile__import__cleanup_stage1();
  END;
  $$ LANGUAGE plpgsql;

--------------------------------------------------------------------------------
-- returns the existing csvlogfiles without showing actual filenames
--   # without showing filenames, this one is suitable to be called by users !!!
--   # this allows to find 'lost' logs, without actually exposing disk/path/file info
CREATE OR REPLACE FUNCTION tlog.dblog__logfile__list() RETURNS SETOF tlog.dblogfiles_imported AS
  $$
    import glob
    import datetime

    _plan_notice = plpy.prepare("SELECT tlog.dblog__raisenotice($1::varchar);", ["varchar"])

    _ret = plpy.execute("SELECT pg_backend_pid() AS pid, current_database() AS db, tlog.dblog__logfile__dir_name__get() AS logdir")

    _current_pid = _ret[0]["pid"]
    _current_db = _ret[0]["db"]
    _LogDir = _ret[0]["logdir"] + '/'
    _LenDir = len(_LogDir)

    # -- csvlog . db-name . date(iso) . time(partitioned/number) . pid . csv
    # -- 0        1         2           3                          4     5
    for _file in glob.glob(_LogDir + "csvlog.*.*.*.*.csv"):
      _filename = _file[_LenDir:]
      _parts = _filename.split(".")
      _partscount = len(_parts)

      _message = "tlog.dblog__logfile__list: " + _filename
      plpy.execute(_plan_notice, [_message])

      if ((_partscount == 6) and (_parts[0] == 'csvlog') and (_parts[5] == 'csv')):
        _database = _parts[1]
        try:
          _pid = int(_parts[4])
        except Exception as _E:
          _pid = 0

        try:
          _date = datetime.datetime.strptime(_parts[2], '%Y%m%d')
        except Exception as _E:
          _pid = 0

        try:
          _time = int(_parts[3])
        except Exception as _E:
          _pid = 0

        if ((_pid > 0)):
          yield(_date, _database, _pid, '', _time)
  $$ LANGUAGE plpython3u;

--------------------------------------------------------------------------------
-- !!! NOT supposed to be called by users directly !!!
-- deletes the temporary importfile
CREATE OR REPLACE FUNCTION tlog.dblog__logfile__client_delete() RETURNS void AS
  $$
    import os
    from tempfile import gettempdir

    _ret = plpy.execute("SELECT pg_backend_pid()::varchar AS pid, tlog.dblog__logfile__dir_name__get()::varchar AS logdir;")
    _current_pid = _ret[0]["pid"]
    _logdir      = _ret[0]["logdir"]

    _logfilename = 'csvlog.client.import.' + str(_current_pid) + '.csv'
    _logfilename = os.path.join(_logdir, _logfilename)

    if os.path.exists(_logfilename):
        os.remove(_logfilename)
  $$ LANGUAGE plpython3u;

--------------------------------------------------------------------------------
-- !!! NOT supposed to be called by users directly !!!
-- writes the temporary importfile
CREATE OR REPLACE FUNCTION tlog.dblog__logfile__client_write(file_data bytea) RETURNS boolean AS
  $$
    import os
    from tempfile import gettempdir

    _ret = plpy.execute("SELECT pg_backend_pid()::varchar AS pid, tlog.dblog__logfile__dir_name__get()::varchar AS logdir;")
    _current_pid = _ret[0]["pid"]
    _logdir      = _ret[0]["logdir"]

    _logfilename = 'csvlog.client.import.' + str(_current_pid) + '.csv'
    _logfilename = os.path.join(_logdir, _logfilename)

    with open(_logfilename, 'wb') as fd:
      fd.write(file_data)

    return True
  $$ LANGUAGE plpython3u;

--------------------------------------------------------------------------------
-- imports a csvlogfile-data as a client-log
--   format of the file is the same as for the dblog (file_data is the binary content of the file)
--   client logs have their own logid's -> rewrite dbl_id, dbl_parent_id
--   client logs might not had an actual db connection -> rewrite dbl_conn_application (include logid from start of the import)
CREATE OR REPLACE FUNCTION tlog.dblog__logfile__client_import(file_data bytea)
  RETURNS bigint
  SET escape_string_warning = false
  AS $$
  DECLARE
    _logID            bigint;
    _logfilename      varchar;
    _tmpTable_import  varchar;
    _tmpTable_rewrite varchar;

    _SQL              varchar;

  BEGIN
    _logID            := tlog.dblog__log_line__create('pltSQL'::tlog.dblog_logtype, 'pllDebugVerbose'::tlog.dblog_loglevel, 'plsDB'::tlog.dblog_logsource, 0, 'tlog.dblog__logfile__client_import', '', 'start');
    _logfilename      := FORMAT('%s/csvlog.client.import.%s.csv',   tlog.dblog__logfile__dir_name__get(), pg_backend_pid()::varchar);
    _tmpTable_import  := FORMAT('dblog_tmp_clientimport_%s',                                              pg_backend_pid()::varchar);
    _tmpTable_rewrite := FORMAT('dblog_tmp_clientimportrewrite_%s',                                       pg_backend_pid()::varchar);

    PERFORM tlog.dblog__log_line__create('pltSQL'::tlog.dblog_logtype, 'pllDebugVerbose'::tlog.dblog_loglevel, 'plsDB'::tlog.dblog_logsource, 0, 'tlog.dblog__logfile__client_import', '', _logfilename);

    BEGIN
      -- write to disk
      PERFORM tlog.dblog__logfile__client_write(file_data);

      -- import to temp table
      EXECUTE FORMAT('DROP TABLE IF EXISTS %s',                            _tmpTable_import);
      EXECUTE FORMAT('CREATE TEMP TABLE %s(dbl_id bigint, logentry json)', _tmpTable_import);
      -- QUOTE e'\x01' DELIMITER e'\x02
      -- This is needed, to allow json-escaped characters inside a json-string
      -- linebreaks are escaped in json-strings with \n
      _SQL := FORMAT('COPY %s FROM ''%s'' CSV QUOTE e''\x01'' DELIMITER e''\x02'';', _tmpTable_import, _logfilename);
      RAISE NOTICE '%', _SQL;
      EXECUTE _SQL;

      -- expand (json) to another temp table
      EXECUTE FORMAT('DROP TABLE IF EXISTS %s', _tmpTable_rewrite);
      _SQL :=
        FORMAT
        (
          $SQL$
            CREATE TEMP TABLE %s AS
            (
              SELECT
                clientlog.dbl_id AS dbl_id_orig,
                clientlog.*
              FROM
                %s
              CROSS JOIN lateral
                json_populate_record(null::tlog.dblog, %s.logentry) AS clientlog
            )
          $SQL$,
          _tmpTable_rewrite, _tmpTable_import, _tmpTable_import
        )
      ;
      RAISE NOTICE '%', _SQL;
      EXECUTE _SQL;

      -- generate propper dblog_id values before copying over to tlog.dblog and mark them as imported with _logID
      _SQL := FORMAT('UPDATE %s SET dbl_id = nextval(''tlog.dblog_id''::regclass), dbl_conn_application = concat(''[%s]'', dbl_conn_application);', _tmpTable_rewrite, _logID::varchar);
      RAISE NOTICE '%', _SQL;
      EXECUTE _SQL;

      -- rewrite dbl_parent_id
      _SQL :=
        FORMAT
        (
          $SQL$
            WITH _src as
            (
              SELECT
                (
                  CASE
                    WHEN (_rec.dbl_parent_id = -1) THEN -1
                  ELSE
                    -1 * _rec.dbl_parent_id
                  END
                ) AS new_parent_id,
                _parentrec.dbl_id,
                _rec.dbl_id AS rec_id
              FROM
                %s AS _rec
              LEFT JOIN
                %s AS _parentrec ON _parentrec.dbl_id_orig = _rec.dbl_parent_id
            )
            UPDATE
              %s
            SET
              dbl_parent_id = _src.new_parent_id
            FROM
              _src
            WHERE
              _src.rec_id = %s.dbl_id;
          $SQL$,
          _tmpTable_rewrite, _tmpTable_rewrite, _tmpTable_rewrite, _tmpTable_rewrite
        )
      ;
      EXECUTE _SQL;

      -- drop column that does not exists in tlog.dblog
      _SQL := FORMAT('ALTER TABLE %s DROP COLUMN dbl_id_orig', _tmpTable_rewrite);
      EXECUTE _SQL;

      -- copy over all imported and prepared logentries
      _SQL := FORMAT('INSERT INTO tlog.dblog SELECT * FROM %s', _tmpTable_rewrite);
      EXECUTE _SQL;

      PERFORM tlog.dblog__logfile__client_delete();

      RAISE NOTICE 'tlog.dblog__logfile__client_import ok';
    EXCEPTION
      WHEN OTHERS THEN
        BEGIN
          -- cleanup the temporary importfile
          PERFORM tlog.dblog__logfile__client_delete();
        EXCEPTION
          WHEN OTHERS THEN
            -- do nothing silent, failing to delete the temporary importfile is not critical
            -- raising here (error or notice) might cause a loop (this should be the bottom-lowest endpoint regarding calls)
            NULL;
        END;
        RAISE; -- re-raise original exception
    END;

    RETURN _logID;
  END;
  $$ LANGUAGE plpgsql SECURITY DEFINER;

--------------------------------------------------------------------------------
-- !!! DO NOT USE FULL PATH FILENAMES AS PARAMETERS !!!
-- !!! DO NOT USE FULL PATH FILENAMES AS PARAMETERS !!!
-- !!! DO NOT USE FULL PATH FILENAMES AS PARAMETERS !!!
--------------------------------------------------------------------------------
-- !!! NOT supposed to be called by users directly !!!
-- write a line to the csvlogfile
--   the encoding of the file-open needs to be the same as the default db-encoding
--   this should be always utf-8, considering the stand of things at moment of coding
CREATE OR REPLACE FUNCTION tlog.dblog__logfile__log_line__write(a_csvline varchar) RETURNS void AS
  $$
    import os
    import sys
    from datetime import timezone
    import datetime

    _datetime = datetime.datetime.now(timezone.utc)
    _datetime = _datetime.replace(tzinfo=None)
    _date = _datetime.date()
    _time = _datetime.time()
    _time_start = datetime.datetime.combine(_date, datetime.datetime.min.time())
    _timedelta = (_datetime - _time_start)
    _time_partitioned = int(((_timedelta.total_seconds() // 60) // 10) * 10)

    if 'invalid_date' not in SD:
      SD['invalid_date'] = datetime.datetime(1, 1, 1, 0, 0, 0)

    if _date != SD.get('date', SD['invalid_date']):
      SD['time_partitioned'] = int(-1)
      SD['date'] = _date
      _plan_notice = plpy.prepare("SELECT tlog.dblog__raisenotice($1::varchar);", ["varchar"])
      plpy.execute(_plan_notice, ['sd.date changed: ' + SD['date'].strftime("%d")])

    if _time_partitioned != SD.get('time_partitioned', int(-1)):
      # -- get filename and sanitize
      _ret = plpy.execute("SELECT tlog.dblog__logfile__file_name__get() AS logfile")
      SD['time_partitioned'] = _time_partitioned
      SD['filename'] = _ret[0]["logfile"]
      SD['filename'] = ''.join(filter( lambda x: x in '.-_0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz', SD['filename'] ))
      _plan_notice = plpy.prepare("SELECT tlog.dblog__raisenotice($1::varchar);", ["varchar"])
      plpy.execute(_plan_notice, ['sd.time_partitioned changed: ' + SD['filename']])

    if not 'directory' in SD:
      # -- SD and GD dictionaries are reset on each session
      # -- so its ok to stick pid to 'directory'
      _ret = plpy.execute("SELECT pg_backend_pid() AS pid, current_database() AS db, tlog.dblog__logfile__dir_name__get() AS logdir")
      SD['pid'] = _ret[0]["pid"]
      SD['db'] = _ret[0]["db"]
      SD['directory'] = _ret[0]["logdir"] + '/'
      _plan_notice = plpy.prepare("SELECT tlog.dblog__raisenotice($1::varchar);", ["varchar"])
      plpy.execute(_plan_notice, ['sd.directory changed: ' + SD['directory']])

    _logfilename = os.path.join(SD['directory'], SD['filename'])

    with open(_logfilename, 'a', encoding='utf8') as fd:
      fd.write(a_csvline + f'\n')

  $$ LANGUAGE plpython3u;


--------------------------------------------------------------------------------
-- !!! NOT supposed to be called by users directly !!!
-- build the csv-line for the logfile and call tlog.dblog__logfile__write_line
CREATE OR REPLACE FUNCTION tlog.dblog__log_line__create(
    a_logtype          tlog.dblog_logtype,
    a_loglevel         tlog.dblog_loglevel,
    a_logsource        tlog.dblog_logsource,
    a_conn_suser       varchar,
    a_conn_cuser       varchar,
    a_conn_ip          inet,
    a_conn_pid         integer,
    a_conn_application varchar,
    a_parent_id        bigint,
    a_func_name        varchar,
    a_ctx              varchar,
    a_msg              varchar
  )
  RETURNS bigint
  SET escape_string_warning = false
  AS $$
  DECLARE
    _logrow tlog.dblog%rowtype;
    _json   json;
    _csv    varchar;

  BEGIN
    _logrow.dbl_id               := nextval('tlog.dblog_id'::regclass);
    _logrow.dbl_ts_tx            := timezone('utc'::text, current_timestamp);
    _logrow.dbl_ts_stmt          := timezone('utc'::text, clock_timestamp());
    _logrow.dbl_type             := a_logtype;
    _logrow.dbl_level            := a_loglevel;
    _logrow.dbl_source           := a_logsource;
    _logrow.dbl_conn_suser       := a_conn_suser;
    _logrow.dbl_conn_cuser       := a_conn_cuser;
    _logrow.dbl_conn_ip          := a_conn_ip;
    _logrow.dbl_conn_pid         := a_conn_pid;
    _logrow.dbl_conn_application := a_conn_application;
    _logrow.dbl_txid             := pg_current_xact_id_if_assigned()::varchar;
    _logrow.dbl_txid_snapshot    := pg_current_snapshot();
    _logrow.dbl_parent_id        := a_parent_id;
    _logrow.dbl_func_name        := a_func_name;
    _logrow.dbl_ctx              := a_ctx;
    _logrow.dbl_msg              := a_msg;

    _json := row_to_json(_logrow);
    -- see: tlog.dblog__logfile__import
    _csv := concat(_logrow.dbl_id::varchar, e'\x02', e'\x01', _json::varchar, e'\x01');
    -- RAISE NOTICE '%', _csv;

    PERFORM tlog.dblog__logfile__log_line__write(_csv);

    -- Im CI Raise Notices für Debug-Meldungen ausgeben. Dann braucht im CI nicht extra das neue Log dafür importiert und extra ausgegeben werden.
    IF a_logtype = 'pltDebugging' AND a_conn_suser = 'docker' THEN
      -- Nur entsprechend dem gesetzten LogLevel.
      IF SessionVar__get_integer( _suffix => 'sessions_loglevel' ) >= array_position( enum_array, a_loglevel::TLOG.DBLOG_LOGLEVEL ) - 1 FROM enum_range( null::TLOG.DBLOG_LOGLEVEL ) AS enum_array THEN
        -- Format: {LogLevel}: (Kontext): [Funktionsname]: Debug-Meldung
        RAISE NOTICE '%', FORMAT( '{%L}: (%L): [%L]: %s', a_loglevel, a_ctx, a_func_name, a_msg );
      END IF;
    END IF;

    RETURN _logrow.dbl_id;
  END $$ LANGUAGE plpgsql;

--------------------------------------------------------------------------------
CREATE OR REPLACE FUNCTION tlog.dblog__log_line__create(
    a_logtype   tlog.dblog_logtype,
    a_loglevel  tlog.dblog_loglevel,
    a_logsource tlog.dblog_logsource,
    a_parent_id bigint,
    a_func_name varchar,
    a_ctx       varchar,
    a_msg       varchar
  )
  RETURNS bigint
  AS $$
    SELECT
      tlog.dblog__log_line__create
      (
        a_logtype, a_loglevel, a_logsource,
        session_user::varchar, current_user::varchar, inet_client_addr(), pg_backend_pid(), current_setting('application_name'), -- session_user, current_user without brackets !!!
        a_parent_id, a_func_name,
        a_ctx, a_msg
      )
  $$ LANGUAGE sql;
---